﻿#region << Using Directives >>
using System;
#endregion

namespace Volpe.Cafe.Generic
{
    /// <summary>
    /// Represents an object that stores compliance modeling data for each regulatory class.
    /// </summary>
    [Serializable]
    public class RCValue<T> : ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="RCValue{T}"/> class.
        /// </summary>
        public RCValue()
        {
            this.Items = new T[Classes.Length];
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCValue{T}"/> class using the specified values.
        /// </summary>
        public RCValue(T passengerCar, T lightTruck, T lightTruck2b3)
        {
            this.Items = new T[] { passengerCar, lightTruck, lightTruck2b3 };
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Determines whether the specified <see cref="RCValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="RCValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool operator ==(RCValue<T> value1, RCValue<T> value2)
        {
            return Equals(value1, value2);
        }
        /// <summary>
        /// Determines whether the specified <see cref="RCValue{T}"/> values are not equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="RCValue{T}"/> values are not equal; false, otherwise.</returns>
        public static bool operator !=(RCValue<T> value1, RCValue<T> value2)
        {
            return !Equals(value1, value2);
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a shallow copy of the current <see cref="RCValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCValue{T}"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a shallow copy of the current <see cref="RCValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCValue{T}"/>.</returns>
        public RCValue<T> Clone()
        {
            RCValue<T> value = new RCValue<T>();
            this.CopyTo(value);
            return value;
        }
        /// <summary>
        /// Copies the members of the current <see cref="RCValue{T}"/> instance into the specified value.
        /// </summary>
        /// <param name="value">The value where to copy the members of the current instance.</param>
        protected void CopyTo(RCValue<T> value)
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                value.Items[i] = this.Items[i];
            }
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of this <see cref="RCValue{T}"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="RCValue{T}"/> instance.</returns>
        public override string ToString()
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString());
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="RCValue{T}"/>.</returns>
        public override int GetHashCode()
        {
            int hash = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                hash = hash ^ this.Items[i].GetHashCode();
            }
            return hash;
        }

        /// <summary>
        /// Determines whether the specified <see cref="Object"/> is equal to the current <see cref="RCValue{T}"/> value.
        /// </summary>
        /// <param name="obj">The <see cref="Object"/> to compare with the current <see cref="RCValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="Object"/> is equal to the current <see cref="RCValue{T}"/> value; false, otherwise.</returns>
        public override bool Equals(object obj)
        {
            return (obj is RCValue<T>) ? this.Equals((RCValue<T>)obj) : base.Equals(obj);
        }
        /// <summary>
        /// Determines whether the specified <see cref="RCValue{T}"/> value is equal to the current <see cref="RCValue{T}"/> value.
        /// </summary>
        /// <param name="value">The <see cref="RCValue{T}"/> value to compare with the current <see cref="RCValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="RCValue{T}"/> value is equal to the current <see cref="RCValue{T}"/> value;
        ///   false, otherwise.</returns>
        public bool Equals(RCValue<T> value)
        {
            return Equals(this, value);
        }
        /// <summary>
        /// Determines whether the specified <see cref="RCValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first <see cref="RCValue{T}"/> value to compare.</param>
        /// <param name="value2">The second <see cref="RCValue{T}"/> value to compare.</param>
        /// <returns>true, if the specified <see cref="RCValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool Equals(RCValue<T> value1, RCValue<T> value2)
        {
            if (object.ReferenceEquals(value1, null) && object.ReferenceEquals(value2, null)) { return true ; }
            if (object.ReferenceEquals(value1, null) || object.ReferenceEquals(value2, null)) { return false; }
            //
            for (int i = 0; i < value1.Items.Length; i++)
            {
                if (!value1.Items[i].Equals(value2.Items[i])) { return false; }
            }
            return true;
        }

        #endregion

        /// <summary>
        /// Resets all members of this <see cref="RCValue{T}"/> instance to their default values (0, false, or null).
        /// </summary>
        public void Clear()
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                this.Items[i] = default(T);
            }
        }

        /// <summary>
        /// Returns the index corresponding to the specified <see cref="RegulatoryClass"/>.
        /// </summary>
        /// <param name="regClass"></param>
        /// <returns>The index corresponding to the specified <see cref="RegulatoryClass"/>.</returns>
        protected int GetIndex(RegulatoryClass regClass)
        {
            return (int)regClass - 1;
            //switch (regClass)
            //{
            //    case RegulatoryClass.LightTruck2b3: return  2;
            //    case RegulatoryClass.LightTruck   : return  1;
            //    case RegulatoryClass.PassengerCar : return  0;
            //    default:                            return -1;
            //}
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets or sets the compliance modeling data value based on the given regulatory class.</summary>
        /// <exception cref="System.IndexOutOfRangeException">The value specified does not represent a valid regulatory class.</exception>
        public T this[RegulatoryClass regClass]
        {
            get { return this.Items[this.GetIndex(regClass)]; }
            set { this.Items[this.GetIndex(regClass)] = value; }
        }

        /// <summary>Gets an array of compliance modeling data specified by <see cref="VehicleClass"/>.</summary>
        public T[] Items { get; private set; }

        ///// <summary>Gets or sets the compliance modeling data for vehicles regulated as passenger cars.</summary>
        //public T PassengerCar { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for vehicles regulated as light trucks.</summary>
        //public T LightTruck { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for vehicles regulated as class-2b/3 light trucks.</summary>
        //public T LightTruck2b3 { get; set; }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies the <see cref="RegulatoryClass"/>es supported by this object. This field is read-only.</summary>
        public static readonly RegulatoryClass[] Classes = new RegulatoryClass[] { RegulatoryClass.PassengerCar,
                                                                                   RegulatoryClass.LightTruck,
                                                                                   RegulatoryClass.LightTruck2b3 };
        /// <summary>Provides the friendly name for each of the supported <see cref="RegulatoryClass"/>es. This field is read-only.</summary>
        public static readonly string[] Names = new string[] { "Passenger Car",
                                                               "Light Truck",
                                                               "Light Truck 2b/3" };

        #endregion

    }
}
